<?php
/**
 * Theme functions and definitions
 *
 * @package HelloElementor
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

define( 'HELLO_ELEMENTOR_VERSION', '3.4.4' );
define( 'EHP_THEME_SLUG', 'hello-elementor' );

define( 'HELLO_THEME_PATH', get_template_directory() );
define( 'HELLO_THEME_URL', get_template_directory_uri() );
define( 'HELLO_THEME_ASSETS_PATH', HELLO_THEME_PATH . '/assets/' );
define( 'HELLO_THEME_ASSETS_URL', HELLO_THEME_URL . '/assets/' );
define( 'HELLO_THEME_SCRIPTS_PATH', HELLO_THEME_ASSETS_PATH . 'js/' );
define( 'HELLO_THEME_SCRIPTS_URL', HELLO_THEME_ASSETS_URL . 'js/' );
define( 'HELLO_THEME_STYLE_PATH', HELLO_THEME_ASSETS_PATH . 'css/' );
define( 'HELLO_THEME_STYLE_URL', HELLO_THEME_ASSETS_URL . 'css/' );
define( 'HELLO_THEME_IMAGES_PATH', HELLO_THEME_ASSETS_PATH . 'images/' );
define( 'HELLO_THEME_IMAGES_URL', HELLO_THEME_ASSETS_URL . 'images/' );

if ( ! isset( $content_width ) ) {
	$content_width = 800; // Pixels.
}

if ( ! function_exists( 'hello_elementor_setup' ) ) {
	/**
	 * Set up theme support.
	 *
	 * @return void
	 */
	function hello_elementor_setup() {
		if ( is_admin() ) {
			hello_maybe_update_theme_version_in_db();
		}

		if ( apply_filters( 'hello_elementor_register_menus', true ) ) {
			register_nav_menus( [ 'menu-1' => esc_html__( 'Header', 'hello-elementor' ) ] );
			register_nav_menus( [ 'menu-2' => esc_html__( 'Footer', 'hello-elementor' ) ] );
		}

		if ( apply_filters( 'hello_elementor_post_type_support', true ) ) {
			add_post_type_support( 'page', 'excerpt' );
		}

		if ( apply_filters( 'hello_elementor_add_theme_support', true ) ) {
			add_theme_support( 'post-thumbnails' );
			add_theme_support( 'automatic-feed-links' );
			add_theme_support( 'title-tag' );
			add_theme_support(
				'html5',
				[
					'search-form',
					'comment-form',
					'comment-list',
					'gallery',
					'caption',
					'script',
					'style',
					'navigation-widgets',
				]
			);
			add_theme_support(
				'custom-logo',
				[
					'height'      => 100,
					'width'       => 350,
					'flex-height' => true,
					'flex-width'  => true,
				]
			);
			add_theme_support( 'align-wide' );
			add_theme_support( 'responsive-embeds' );

			/*
			 * Editor Styles
			 */
			add_theme_support( 'editor-styles' );
			add_editor_style( 'editor-styles.css' );

			/*
			 * WooCommerce.
			 */
			if ( apply_filters( 'hello_elementor_add_woocommerce_support', true ) ) {
				// WooCommerce in general.
				add_theme_support( 'woocommerce' );
				// Enabling WooCommerce product gallery features (are off by default since WC 3.0.0).
				// zoom.
				add_theme_support( 'wc-product-gallery-zoom' );
				// lightbox.
				add_theme_support( 'wc-product-gallery-lightbox' );
				// swipe.
				add_theme_support( 'wc-product-gallery-slider' );
			}
		}
	}
}
add_action( 'after_setup_theme', 'hello_elementor_setup' );

function hello_maybe_update_theme_version_in_db() {
	$theme_version_option_name = 'hello_theme_version';
	// The theme version saved in the database.
	$hello_theme_db_version = get_option( $theme_version_option_name );

	// If the 'hello_theme_version' option does not exist in the DB, or the version needs to be updated, do the update.
	if ( ! $hello_theme_db_version || version_compare( $hello_theme_db_version, HELLO_ELEMENTOR_VERSION, '<' ) ) {
		update_option( $theme_version_option_name, HELLO_ELEMENTOR_VERSION );
	}
}

if ( ! function_exists( 'hello_elementor_display_header_footer' ) ) {
	/**
	 * Check whether to display header footer.
	 *
	 * @return bool
	 */
	function hello_elementor_display_header_footer() {
		$hello_elementor_header_footer = true;

		return apply_filters( 'hello_elementor_header_footer', $hello_elementor_header_footer );
	}
}

if ( ! function_exists( 'hello_elementor_scripts_styles' ) ) {
	/**
	 * Theme Scripts & Styles.
	 *
	 * @return void
	 */
	function hello_elementor_scripts_styles() {
		if ( apply_filters( 'hello_elementor_enqueue_style', true ) ) {
			wp_enqueue_style(
				'hello-elementor',
				HELLO_THEME_STYLE_URL . 'reset.css',
				[],
				HELLO_ELEMENTOR_VERSION
			);
		}

		if ( apply_filters( 'hello_elementor_enqueue_theme_style', true ) ) {
			wp_enqueue_style(
				'hello-elementor-theme-style',
				HELLO_THEME_STYLE_URL . 'theme.css',
				[],
				HELLO_ELEMENTOR_VERSION
			);
		}

		if ( hello_elementor_display_header_footer() ) {
			wp_enqueue_style(
				'hello-elementor-header-footer',
				HELLO_THEME_STYLE_URL . 'header-footer.css',
				[],
				HELLO_ELEMENTOR_VERSION
			);
		}

		wp_enqueue_style(
			'hello-elementor-header-footer',
			HELLO_THEME_STYLE_URL . 'header-footer.css',
			[],
			HELLO_ELEMENTOR_VERSION
		);
	}
}
add_action( 'wp_enqueue_scripts', 'hello_elementor_scripts_styles' );

if ( ! function_exists( 'hello_elementor_register_elementor_locations' ) ) {
	/**
	 * Register Elementor Locations.
	 *
	 * @param ElementorPro\Modules\ThemeBuilder\Classes\Locations_Manager $elementor_theme_manager theme manager.
	 *
	 * @return void
	 */
	function hello_elementor_register_elementor_locations( $elementor_theme_manager ) {
		if ( apply_filters( 'hello_elementor_register_elementor_locations', true ) ) {
			$elementor_theme_manager->register_all_core_location();
		}
	}
}
add_action( 'elementor/theme/register_locations', 'hello_elementor_register_elementor_locations' );

if ( ! function_exists( 'hello_elementor_content_width' ) ) {
	/**
	 * Set default content width.
	 *
	 * @return void
	 */
	function hello_elementor_content_width() {
		$GLOBALS['content_width'] = apply_filters( 'hello_elementor_content_width', 800 );
	}
}
add_action( 'after_setup_theme', 'hello_elementor_content_width', 0 );

if ( ! function_exists( 'hello_elementor_add_description_meta_tag' ) ) {
	/**
	 * Add description meta tag with excerpt text.
	 *
	 * @return void
	 */
	function hello_elementor_add_description_meta_tag() {
		if ( ! apply_filters( 'hello_elementor_description_meta_tag', true ) ) {
			return;
		}

		if ( ! is_singular() ) {
			return;
		}

		$post = get_queried_object();
		if ( empty( $post->post_excerpt ) ) {
			return;
		}

		echo '<meta name="description" content="' . esc_attr( wp_strip_all_tags( $post->post_excerpt ) ) . '">' . "\n";
	}
}
add_action( 'wp_head', 'hello_elementor_add_description_meta_tag' );

// Settings page
require get_template_directory() . '/includes/settings-functions.php';

// Header & footer styling option, inside Elementor
require get_template_directory() . '/includes/elementor-functions.php';

if ( ! function_exists( 'hello_elementor_customizer' ) ) {
	// Customizer controls
	function hello_elementor_customizer() {
		if ( ! is_customize_preview() ) {
			return;
		}

		if ( ! hello_elementor_display_header_footer() ) {
			return;
		}

		require get_template_directory() . '/includes/customizer-functions.php';
	}
}
add_action( 'init', 'hello_elementor_customizer' );

if ( ! function_exists( 'hello_elementor_check_hide_title' ) ) {
	/**
	 * Check whether to display the page title.
	 *
	 * @param bool $val default value.
	 *
	 * @return bool
	 */
	function hello_elementor_check_hide_title( $val ) {
		if ( defined( 'ELEMENTOR_VERSION' ) ) {
			$current_doc = Elementor\Plugin::instance()->documents->get( get_the_ID() );
			if ( $current_doc && 'yes' === $current_doc->get_settings( 'hide_title' ) ) {
				$val = false;
			}
		}
		return $val;
	}
}
add_filter( 'hello_elementor_page_title', 'hello_elementor_check_hide_title' );

/**
 * BC:
 * In v2.7.0 the theme removed the `hello_elementor_body_open()` from `header.php` replacing it with `wp_body_open()`.
 * The following code prevents fatal errors in child themes that still use this function.
 */
if ( ! function_exists( 'hello_elementor_body_open' ) ) {
	function hello_elementor_body_open() {
		wp_body_open();
	}
}

add_action('template_redirect', function () {
    if (is_admin()) return;

    // List of unwanted parameters
    $remove_params = ['OrderBy', 'Limit', 'SavedSearch', 'pg', 'p', 'n', 'm'];

    // Current query params
    $current_params = $_GET;
    $clean_params = $current_params;

    $changed = false;

    foreach ($remove_params as $param) {
        if (isset($clean_params[$param])) {
            unset($clean_params[$param]);
            $changed = true;
        }
    }

    if ($changed) {
        $current_url_path = strtok($_SERVER["REQUEST_URI"], '?'); // Just the path, no query string
        $clean_url = $current_url_path;

        if (!empty($clean_params)) {
            $clean_url .= '?' . http_build_query($clean_params);
        }

        wp_redirect($clean_url, 301);
        exit;
    }
});

add_filter('posts_orderby', function($orderby, $query) {
    if (
        !is_admin() && 
        $query->get('post_type') === 'agent' && 
        isset($query->query_vars['orderby']) && 
        $query->query_vars['orderby'] === 'last_slug'
    ) {
        global $wpdb;
        // Sort by the last part of the slug after the last dash
        $orderby = "LOWER(SUBSTRING_INDEX({$wpdb->posts}.post_name, '-', -1)) ASC";
    }
    return $orderby;
}, 10, 2);

function agent_slider_shortcode($atts) {
    ob_start();
    $atts = shortcode_atts(array(
        'limit'    => -1,
        'featured' => '' // comma-separated IDs of featured agents
    ), $atts, 'agent_slider');

    $featured_ids = array_filter(array_map('intval', explode(',', $atts['featured'])));
	
    // Step 1: Query all agent posts
    $args = array(
        'post_type'      => 'agent',
        'posts_per_page' => intval($atts['limit']),
        'orderby'        => 'menu_order',
        'order'          => 'ASC',
        'post_status'    => 'publish',
    );

    $query = new WP_Query($args);

    if ($query->have_posts()) {
        $all_posts = $query->posts;

        // Separate featured and remaining posts
        $featured_posts = [];
        $remaining_posts = [];

        foreach ($all_posts as $post) {
            if (in_array($post->ID, $featured_ids)) {
                $featured_posts[] = $post;
            } else {
                $remaining_posts[] = $post;
            }
        }

        // Sort remaining posts alphabetically by last name
        usort($remaining_posts, function($a, $b) {
            $a_last = strtolower(array_slice(explode(' ', trim(get_the_title($a))), -1)[0]);
            $b_last = strtolower(array_slice(explode(' ', trim(get_the_title($b))), -1)[0]);
            return strcmp($a_last, $b_last);
        });

        // Merge featured posts (in given order) + sorted remaining
        $sorted_posts = array_merge(
            array_map(function($id) use ($featured_posts) {
                foreach ($featured_posts as $p) {
                    if ($p->ID == $id) return $p;
                }
            }, $featured_ids),
            $remaining_posts
        );

        echo '<div class="team_grid">';
        foreach ($sorted_posts as $post) {
            if (!$post) continue;
            setup_postdata($post);
            $specialty = get_post_meta($post->ID, 'specialty', true);
            $permalink = get_permalink($post->ID);
            ?>
            <div class="agent-card relative overflow-hidden rounded-2xl shadow-lg group cursor-pointer">
                <a href="<?php echo esc_url($permalink); ?>">
                    <?php if (has_post_thumbnail($post->ID)) :
                        echo get_the_post_thumbnail($post->ID, 'large', array(
                            'class' => 'w-full h-[500px] object-cover transition-transform duration-300 group-hover:scale-105'
                        ));
                    endif; ?>
                </a>
                <div class="team_info absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black/70 to-transparent p-6 text-white">
                    <h3 class="text-xl font-bold mb-1"><?php echo esc_html(get_the_title($post)); ?></h3>
                </div>
            </div>
            <?php
        }
        echo '</div>';

        wp_reset_postdata();
    }

    return ob_get_clean();
}

add_shortcode('agent_slider', 'agent_slider_shortcode');

function enqueue_custom_assets_for_flexmls_slider() {
    // JavaScript
    wp_enqueue_script(
        'custom-js',
        HELLO_THEME_SCRIPTS_URL . 'custom.js',
        [],
        HELLO_ELEMENTOR_VERSION,
        true
    );

    wp_enqueue_script(
        'slick-js',
        HELLO_THEME_SCRIPTS_URL . 'slick.min.js',
        [],
        HELLO_ELEMENTOR_VERSION,
        true
    );

    wp_enqueue_style(
        'slick-css',
        HELLO_THEME_STYLE_URL . 'slick.min.css',
        [],
        HELLO_ELEMENTOR_VERSION
    );

    wp_enqueue_style(
        'slick-theme-css',
        HELLO_THEME_STYLE_URL . 'slick-theme.css',
        [],
        HELLO_ELEMENTOR_VERSION
    );
	 wp_enqueue_style(
        'helloword-google-fonts',
        'https://fonts.googleapis.com/css2?family=Inter:ital,opsz,wght@0,14..32,100..900;1,14..32,100..900&display=swap',
        false
    );
}
add_action('wp_enqueue_scripts', 'enqueue_custom_assets_for_flexmls_slider');


require HELLO_THEME_PATH . '/theme.php';

HelloTheme\Theme::instance();

function testimonials_post_slider_shortcode($atts) {
$current_post_id = get_the_ID();
    $args = [
        'post_type' => 'testimonials',
        'posts_per_page' => -1 ,
        'orderby' => 'date',
        'order' => 'DESC',
		'meta_query' => [
			[
				'key'     => 'select_agent', 
				'value'   => '"' . $current_post_id . '"',
				'compare' => 'LIKE'
			]
		]
    ];

    $query = new WP_Query($args);

    ob_start();
	 if ($query->have_posts()) { ?>
<div id="testimonial_section">
	<div class="testimonials_container">
		<h2 class="testmonial_title"> Client Testimonials </h2>
		<p class="testmonial_subtitle">What my clients say about their experience </p> 
		<div class="testimonials_slider_wrapper">
			<div class="testimonials_slider">
				<?php while ($query->have_posts()) : $query->the_post(); ?>
				<div class="slide_item rounted_bg h_100">
					<div class="item">
						<?php if (has_post_thumbnail()) : ?>
						<div class="slide_image"><?php the_post_thumbnail('thumbnail'); ?></div>
						<?php endif; ?>
						<div class="authore_info">
							<h3 class="slide_title"><?php the_title(); ?></h3>
							<p class="destination"><?php echo get_field('destination'); ?></p>
							<div class="star_rating">  
								<i class="fa fa-star"></i>
								<i class="fa fa-star"></i>
								<i class="fa fa-star"></i>
								<i class="fa fa-star"></i>
								<i class="fa fa-star"></i>
							</div>
						</div>
					</div>
					<div>
						<p class="testimonial_content"> <?php echo  the_content() ; ?> </p>
					</div>
				</div>
				<?php endwhile; ?>
			</div>
		</div>
	</div>
	</div>
    <?php }
    wp_reset_postdata();
    return ob_get_clean();
}
add_shortcode('testimonials_post_slider', 'testimonials_post_slider_shortcode');

function elementor_random_text() {
    $texts = [
        "Discover luxury properties in New England's most prestigious neighborhoods",
        "Your Dream Home Won't Find Itself",
        "From House Hunting to House Haunting",
        "Where 'Just Looking' Becomes 'Just Bought'",
        "Find Your Sanctuary (Pants Optional)",
        "Turn Your Pinterest Board Into an Address",
        "Less Renting Drama, More House Karma",
        "Your Future Neighbors Are Waiting",
        "Stop Making Your Landlord Rich",
        "Find Your Forever Netflix & Chill Spot",
        "Home is Where You Can Sing in the Shower",
        "Your Next Chapter Needs the Right Setting",
        "From Apartment Therapy to House Therapy",
        "Where 'Location, Location, Location' Meets Reality"
    ];

    return $texts[array_rand($texts)];
}
add_shortcode('banner_random_text', 'elementor_random_text');
