<?php
/**
 * Elite Northeast Realty Theme Functions
 *
 * @package Elite_Northeast_Realty
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Theme setup
 */
function elite_northeast_realty_setup() {
    // Add default posts and comments RSS feed links to head
    add_theme_support( 'automatic-feed-links' );

    // Let WordPress manage the document title
    add_theme_support( 'title-tag' );

    // Enable support for Post Thumbnails on posts and pages
    add_theme_support( 'post-thumbnails' );

    // Set default thumbnail size
    set_post_thumbnail_size( 1200, 800, true );

    // Add custom image sizes
    add_image_size( 'property-featured', 1200, 800, true );
    add_image_size( 'property-thumbnail', 400, 300, true );
    add_image_size( 'agent-photo', 300, 400, true );

    // Register navigation menus
    register_nav_menus( array(
        'primary' => esc_html__( 'Primary Menu', 'elite-northeast-realty' ),
        'mobile'  => esc_html__( 'Mobile Menu', 'elite-northeast-realty' ),
        'footer-properties' => esc_html__( 'Footer Properties Menu', 'elite-northeast-realty' ),
        'footer-company'  => esc_html__( 'Footer Company Menu', 'elite-northeast-realty' ),
    ) );

    // Switch default core markup to output valid HTML5
    add_theme_support( 'html5', array(
        'search-form',
        'comment-form',
        'comment-list',
        'gallery',
        'caption',
        'style',
        'script',
    ) );

    // Add support for custom logo
    add_theme_support( 'custom-logo', array(
        'height'      => 100,
        'width'       => 400,
        'flex-height' => true,
        'flex-width'  => true,
        'header-text' => array( 'site-title', 'site-description' ),
    ) );

    // Add support for responsive embeds
    add_theme_support( 'responsive-embeds' );

    // Add support for wide alignment
    add_theme_support( 'align-wide' );
}
add_action( 'after_setup_theme', 'elite_northeast_realty_setup' );

/**
 * Enqueue scripts and styles
 */
function elite_northeast_realty_scripts() {
    // Enqueue theme stylesheet
    wp_enqueue_style( 
        'elite-northeast-realty-style', 
        get_stylesheet_uri(), 
        array(), 
        wp_get_theme()->get( 'Version' ) 
    );

    // Enqueue Tailwind CSS from CDN
    wp_enqueue_script(
        'tailwindcss',
        'https://cdn.tailwindcss.com/3.4.16',
        array(),
        '3.4.16',
        false
    );
    
    // Add Tailwind config inline
    wp_add_inline_script('tailwindcss', '
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: "#2563eb",
                        secondary: "#64748b"
                    },
                    borderRadius: {
                        "none": "0px",
                        "sm": "4px",
                        DEFAULT: "8px",
                        "md": "12px",
                        "lg": "16px",
                        "xl": "20px",
                        "2xl": "24px",
                        "3xl": "32px",
                        "full": "9999px",
                        "button": "8px"
                    }
                }
            }
        }
    ');
    
    // Enqueue Google Fonts
    wp_enqueue_style(
        'google-fonts',
        'https://fonts.googleapis.com/css2?family=Pacifico&family=Inter:wght@300;400;500;600;700&display=swap',
        array(),
        null
    );
    
    // Enqueue Remix Icons
    wp_enqueue_style(
        'remixicon',
        'https://cdnjs.cloudflare.com/ajax/libs/remixicon/4.6.0/remixicon.min.css',
        array(),
        '4.6.0'
    );
    
    // Enqueue custom styles
    wp_enqueue_style( 
        'elite-northeast-realty-custom', 
        get_template_directory_uri() . '/assets/css/custom.css', 
        array(), 
        wp_get_theme()->get( 'Version' ) 
    );
    
    // Enqueue custom scripts
    wp_enqueue_script(
        'elite-northeast-realty-navigation',
        get_template_directory_uri() . '/assets/js/navigation.js',
        array(),
        wp_get_theme()->get( 'Version' ),
        true
    );
    
    // Enqueue front page specific styles and scripts
    if ( is_front_page() ) {
        wp_enqueue_style( 
            'elite-northeast-realty-front-page', 
            get_template_directory_uri() . '/assets/css/front-page.css', 
            array( 'elite-northeast-realty-custom' ), 
            wp_get_theme()->get( 'Version' ) 
        );
        
        wp_enqueue_script(
            'elite-northeast-realty-front-page',
            get_template_directory_uri() . '/assets/js/front-page.js',
            array(),
            wp_get_theme()->get( 'Version' ),
            true
        );
    }
    
    // Enqueue about page specific styles
    if ( is_page_template( 'page-about.php' ) ) {
        wp_enqueue_style( 
            'elite-northeast-realty-about-page', 
            get_template_directory_uri() . '/assets/css/about-page.css', 
            array( 'elite-northeast-realty-custom' ), 
            wp_get_theme()->get( 'Version' ) 
        );
    }
    
    // Enqueue property single page specific styles and scripts
    if ( is_singular( 'property' ) ) {
        wp_enqueue_style( 
            'elite-northeast-realty-property-single', 
            get_template_directory_uri() . '/assets/css/property-single.css', 
            array( 'elite-northeast-realty-custom' ), 
            wp_get_theme()->get( 'Version' ) 
        );
        
        wp_enqueue_script(
            'elite-northeast-realty-property-single',
            get_template_directory_uri() . '/assets/js/property-single.js',
            array( 'jquery' ),
            wp_get_theme()->get( 'Version' ),
            true
        );
        
        // Localize script for AJAX
        wp_localize_script( 'elite-northeast-realty-property-single', 'propertyAjax', array(
            'ajaxurl' => admin_url( 'admin-ajax.php' ),
            'nonce'   => wp_create_nonce( 'property_form_nonce' )
        ) );
    }
    
    // Enqueue agent single page specific styles and scripts
    if ( is_singular( 'agent' ) ) {
        wp_enqueue_style( 
            'elite-northeast-realty-agent-single', 
            get_template_directory_uri() . '/assets/css/agent-single.css', 
            array( 'elite-northeast-realty-custom' ), 
            wp_get_theme()->get( 'Version' ) 
        );
        
        wp_enqueue_script(
            'elite-northeast-realty-agent-single',
            get_template_directory_uri() . '/assets/js/agent-single.js',
            array( 'jquery' ),
            wp_get_theme()->get( 'Version' ),
            true
        );
        
        // Localize script for AJAX
        wp_localize_script( 'elite-northeast-realty-agent-single', 'elite_realty_ajax', array(
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'template_url' => get_template_directory_uri(),
            'nonce'   => wp_create_nonce( 'agent_contact_nonce' )
        ) );
    }
    
    // Enqueue community archive page specific styles and scripts
    if ( is_post_type_archive( 'community' ) || is_tax( 'state' ) ) {
        wp_enqueue_style( 
            'elite-northeast-realty-archive-community', 
            get_template_directory_uri() . '/assets/css/archive-community.css', 
            array( 'elite-northeast-realty-custom' ), 
            wp_get_theme()->get( 'Version' ) 
        );
        
        wp_enqueue_script(
            'elite-northeast-realty-archive-community',
            get_template_directory_uri() . '/assets/js/archive-community.js',
            array(),
            wp_get_theme()->get( 'Version' ),
            true
        );
    }

    // Enqueue comment reply script when needed
    if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
        wp_enqueue_script( 'comment-reply' );
    }
}
add_action( 'wp_enqueue_scripts', 'elite_northeast_realty_scripts' );

/**
 * Register widget areas
 */
function elite_northeast_realty_widgets_init() {
    register_sidebar( array(
        'name'          => esc_html__( 'Sidebar', 'elite-northeast-realty' ),
        'id'            => 'sidebar-1',
        'description'   => esc_html__( 'Add widgets here to appear in your sidebar.', 'elite-northeast-realty' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s p-6 mb-6 bg-white rounded-lg shadow">',
        'after_widget'  => '</section>',
        'before_title'  => '<h2 class="widget-title text-xl font-bold mb-4">',
        'after_title'   => '</h2>',
    ) );

    register_sidebar( array(
        'name'          => esc_html__( 'Footer Column 1', 'elite-northeast-realty' ),
        'id'            => 'footer-1',
        'description'   => esc_html__( 'Add widgets here to appear in the first footer column.', 'elite-northeast-realty' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s mb-6">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widget-title text-lg font-bold mb-3">',
        'after_title'   => '</h3>',
    ) );

    register_sidebar( array(
        'name'          => esc_html__( 'Footer Column 2', 'elite-northeast-realty' ),
        'id'            => 'footer-2',
        'description'   => esc_html__( 'Add widgets here to appear in the second footer column.', 'elite-northeast-realty' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s mb-6">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widget-title text-lg font-bold mb-3">',
        'after_title'   => '</h3>',
    ) );

    register_sidebar( array(
        'name'          => esc_html__( 'Footer Column 3', 'elite-northeast-realty' ),
        'id'            => 'footer-3',
        'description'   => esc_html__( 'Add widgets here to appear in the third footer column.', 'elite-northeast-realty' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s mb-6">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widget-title text-lg font-bold mb-3">',
        'after_title'   => '</h3>',
    ) );

    register_sidebar( array(
        'name'          => esc_html__( 'Footer Column 4', 'elite-northeast-realty' ),
        'id'            => 'footer-4',
        'description'   => esc_html__( 'Add widgets here to appear in the fourth footer column.', 'elite-northeast-realty' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s mb-6">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widget-title text-lg font-bold mb-3">',
        'after_title'   => '</h3>',
    ) );
}
add_action( 'widgets_init', 'elite_northeast_realty_widgets_init' );

/**
 * Custom excerpt length
 */
function elite_northeast_realty_excerpt_length( $length ) {
    return 30;
}
add_filter( 'excerpt_length', 'elite_northeast_realty_excerpt_length' );

/**
 * Custom excerpt more string
 */
function elite_northeast_realty_excerpt_more( $more ) {
    return '...';
}
add_filter( 'excerpt_more', 'elite_northeast_realty_excerpt_more' );

/**
 * Add custom classes to body tag
 */
function elite_northeast_realty_body_classes( $classes ) {
    // Add class for singular pages
    if ( is_singular() ) {
        $classes[] = 'singular';
    }

    // Add class for home page
    if ( is_home() || is_front_page() ) {
        $classes[] = 'home';
    }

    return $classes;
}
add_filter( 'body_class', 'elite_northeast_realty_body_classes' );

/**
 * Custom Nav Walker for primary menu
 */
require get_template_directory() . '/inc/class-elite-realty-nav-walker.php';

/**
 * Customizer additions
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Custom Post Types
 */
require get_template_directory() . '/inc/custom-post-types.php';

/**
 * Fallback menu function
 */
function elite_realty_footer_menu_fallback() {
    $fallback_menu = array(
        'Properties' => array(
            'Buy' => '#',
            'Sell' => '#',
            'Rent' => '#',
            'Commercial' => '#',
            'Communities' => '#'
        ),
        'Company' => array(
            'About Us' => '#',
            'Our Team' => '#',
            'Careers' => '#',
            'Contact' => '#'
        )
    );
    
    echo '<ul class="space-y-2 text-gray-400">';
    foreach ($fallback_menu['Properties'] as $label => $url) {
        echo '<li><a href="' . esc_url($url) . '" class="hover:text-white transition-colors">' . esc_html($label) . '</a></li>';
    }
    echo '</ul>';
}

/**
 * Handle property form submissions via AJAX
 */
function elite_northeast_realty_handle_property_form() {
    // Verify nonce
    if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'property_form_nonce' ) ) {
        wp_send_json_error( 'Invalid security token.' );
    }
    
    // Get form type and data
    $form_type = isset( $_POST['form_type'] ) ? sanitize_text_field( $_POST['form_type'] ) : '';
    parse_str( $_POST['form_data'], $form_data );
    
    // Sanitize form data
    $sanitized_data = array();
    foreach ( $form_data as $key => $value ) {
        if ( is_array( $value ) ) {
            $sanitized_data[$key] = array_map( 'sanitize_text_field', $value );
        } else {
            $sanitized_data[$key] = sanitize_text_field( $value );
        }
    }
    
    // Get property information
    $property_id = isset( $sanitized_data['property_id'] ) ? intval( $sanitized_data['property_id'] ) : 0;
    $property_title = isset( $sanitized_data['property_title'] ) ? $sanitized_data['property_title'] : '';
    
    // Build email content based on form type
    $subject = '';
    $message = '';
    
    switch ( $form_type ) {
        case 'tour-request':
            $subject = 'New Tour Request for ' . $property_title;
            $message = "New tour request received:\n\n";
            $message .= "Property: " . $property_title . "\n";
            $message .= "Name: " . $sanitized_data['name'] . "\n";
            $message .= "Email: " . $sanitized_data['email'] . "\n";
            $message .= "Phone: " . $sanitized_data['phone'] . "\n";
            $message .= "Preferred Date: " . $sanitized_data['date'] . "\n";
            if ( ! empty( $sanitized_data['message'] ) ) {
                $message .= "Message: " . $sanitized_data['message'] . "\n";
            }
            break;
            
        case 'contact-agent':
            $subject = 'New Inquiry for ' . $property_title;
            $message = "New property inquiry received:\n\n";
            $message .= "Property: " . $property_title . "\n";
            $message .= "Name: " . $sanitized_data['name'] . "\n";
            $message .= "Email: " . $sanitized_data['email'] . "\n";
            $message .= "Phone: " . $sanitized_data['phone'] . "\n";
            $message .= "Message: " . $sanitized_data['message'] . "\n";
            break;
            
        case 'request-info':
            $subject = 'Information Request for ' . $property_title;
            $message = "New information request received:\n\n";
            $message .= "Property: " . $property_title . "\n";
            $message .= "Name: " . $sanitized_data['name'] . "\n";
            $message .= "Email: " . $sanitized_data['email'] . "\n";
            if ( ! empty( $sanitized_data['phone'] ) ) {
                $message .= "Phone: " . $sanitized_data['phone'] . "\n";
            }
            if ( ! empty( $sanitized_data['info_type'] ) ) {
                $message .= "Information Requested: " . implode( ', ', $sanitized_data['info_type'] ) . "\n";
            }
            if ( ! empty( $sanitized_data['message'] ) ) {
                $message .= "Additional Details: " . $sanitized_data['message'] . "\n";
            }
            break;
            
        default:
            wp_send_json_error( 'Invalid form type.' );
    }
    
    // Get admin email
    $admin_email = get_option( 'admin_email' );
    
    // Send email
    $headers = array(
        'Content-Type: text/plain; charset=UTF-8',
        'From: ' . get_bloginfo( 'name' ) . ' <' . $admin_email . '>',
        'Reply-To: ' . $sanitized_data['email']
    );
    
    $sent = wp_mail( $admin_email, $subject, $message, $headers );
    
    if ( $sent ) {
        // Optionally save form submission to database
        $submission_data = array(
            'post_title'   => $subject,
            'post_content' => $message,
            'post_status'  => 'private',
            'post_type'    => 'form_submission',
            'meta_input'   => array(
                'form_type'     => $form_type,
                'property_id'   => $property_id,
                'submitter_name' => $sanitized_data['name'],
                'submitter_email' => $sanitized_data['email']
            )
        );
        
        // Only save if form_submission post type exists
        if ( post_type_exists( 'form_submission' ) ) {
            wp_insert_post( $submission_data );
        }
        
        wp_send_json_success( 'Form submitted successfully!' );
    } else {
        wp_send_json_error( 'Failed to send email. Please try again.' );
    }
}
add_action( 'wp_ajax_submit_property_form', 'elite_northeast_realty_handle_property_form' );
add_action( 'wp_ajax_nopriv_submit_property_form', 'elite_northeast_realty_handle_property_form' );

/**
 * Handle agent contact form submissions
 */
function elite_northeast_realty_handle_agent_contact() {
    // Verify nonce
    if ( ! isset( $_POST['agent_contact_nonce'] ) || ! wp_verify_nonce( $_POST['agent_contact_nonce'], 'agent_contact_form' ) ) {
        wp_send_json_error( 'Invalid security token.' );
    }
    
    // Sanitize form data
    $first_name = sanitize_text_field( $_POST['first_name'] );
    $last_name = sanitize_text_field( $_POST['last_name'] );
    $email = sanitize_email( $_POST['email'] );
    $phone = sanitize_text_field( $_POST['phone'] );
    $message = sanitize_textarea_field( $_POST['message'] );
    $agent_id = intval( $_POST['agent_id'] );
    $agent_name = sanitize_text_field( $_POST['agent_name'] );
    $agent_email = sanitize_email( $_POST['agent_email'] );
    
    // Validate required fields
    if ( empty( $first_name ) || empty( $last_name ) || empty( $email ) || empty( $message ) ) {
        wp_send_json_error( 'Please fill in all required fields.' );
    }
    
    // Build email content
    $subject = 'New Contact Request for ' . $agent_name;
    $email_message = "New contact request received:\n\n";
    $email_message .= "Agent: " . $agent_name . "\n\n";
    $email_message .= "From: " . $first_name . " " . $last_name . "\n";
    $email_message .= "Email: " . $email . "\n";
    $email_message .= "Phone: " . $phone . "\n\n";
    $email_message .= "Message:\n" . $message . "\n";
    
    // Email headers
    $headers = array(
        'Content-Type: text/plain; charset=UTF-8',
        'From: ' . get_bloginfo( 'name' ) . ' <' . get_option( 'admin_email' ) . '>',
        'Reply-To: ' . $first_name . ' ' . $last_name . ' <' . $email . '>'
    );
    
    // Send email to agent
    $sent_to_agent = false;
    if ( ! empty( $agent_email ) ) {
        $sent_to_agent = wp_mail( $agent_email, $subject, $email_message, $headers );
    }
    
    // Also send to admin as backup
    $admin_email = get_option( 'admin_email' );
    $sent_to_admin = wp_mail( $admin_email, $subject, $email_message, $headers );
    
    if ( $sent_to_agent || $sent_to_admin ) {
        // Optionally save form submission
        $submission_data = array(
            'post_title'   => $subject,
            'post_content' => $email_message,
            'post_status'  => 'private',
            'post_type'    => 'agent_inquiry',
            'meta_input'   => array(
                'agent_id'       => $agent_id,
                'agent_name'     => $agent_name,
                'submitter_name' => $first_name . ' ' . $last_name,
                'submitter_email' => $email,
                'submitter_phone' => $phone
            )
        );
        
        // Only save if agent_inquiry post type exists
        if ( post_type_exists( 'agent_inquiry' ) ) {
            wp_insert_post( $submission_data );
        }
        
        wp_send_json_success( 'Thank you for your message. The agent will contact you soon!' );
    } else {
        wp_send_json_error( 'Failed to send message. Please try again or contact us directly.' );
    }
}
add_action( 'wp_ajax_handle_agent_contact', 'elite_northeast_realty_handle_agent_contact' );
add_action( 'wp_ajax_nopriv_handle_agent_contact', 'elite_northeast_realty_handle_agent_contact' );

// Also handle the form if submitted via admin-post.php (non-AJAX fallback)
function elite_northeast_realty_handle_agent_contact_post() {
    // Verify nonce
    if ( ! isset( $_POST['agent_contact_nonce'] ) || ! wp_verify_nonce( $_POST['agent_contact_nonce'], 'agent_contact_form' ) ) {
        wp_die( 'Invalid security token.' );
    }
    
    // Process form (same as AJAX handler)
    $first_name = sanitize_text_field( $_POST['first_name'] );
    $last_name = sanitize_text_field( $_POST['last_name'] );
    $email = sanitize_email( $_POST['email'] );
    $phone = sanitize_text_field( $_POST['phone'] );
    $message = sanitize_textarea_field( $_POST['message'] );
    $agent_id = intval( $_POST['agent_id'] );
    $agent_name = sanitize_text_field( $_POST['agent_name'] );
    $agent_email = sanitize_email( $_POST['agent_email'] );
    
    // Build email content
    $subject = 'New Contact Request for ' . $agent_name;
    $email_message = "New contact request received:\n\n";
    $email_message .= "Agent: " . $agent_name . "\n\n";
    $email_message .= "From: " . $first_name . " " . $last_name . "\n";
    $email_message .= "Email: " . $email . "\n";
    $email_message .= "Phone: " . $phone . "\n\n";
    $email_message .= "Message:\n" . $message . "\n";
    
    // Email headers
    $headers = array(
        'Content-Type: text/plain; charset=UTF-8',
        'From: ' . get_bloginfo( 'name' ) . ' <' . get_option( 'admin_email' ) . '>',
        'Reply-To: ' . $first_name . ' ' . $last_name . ' <' . $email . '>'
    );
    
    // Send emails
    $sent = false;
    if ( ! empty( $agent_email ) ) {
        $sent = wp_mail( $agent_email, $subject, $email_message, $headers );
    }
    
    // Redirect back with status
    if ( $sent ) {
        wp_safe_redirect( add_query_arg( 'agent_contact', 'success', wp_get_referer() ) );
    } else {
        wp_safe_redirect( add_query_arg( 'agent_contact', 'error', wp_get_referer() ) );
    }
    exit;
}
add_action( 'admin_post_handle_agent_contact', 'elite_northeast_realty_handle_agent_contact_post' );
add_action( 'admin_post_nopriv_handle_agent_contact', 'elite_northeast_realty_handle_agent_contact_post' );