<?php
/**
 * Archive Template for Communities
 *
 * @package Elite_Northeast_Realty
 */

get_header();

// Get current state filter if set
$current_state = isset($_GET['state']) ? sanitize_text_field($_GET['state']) : '';
$search_query = isset($_GET['community_search']) ? sanitize_text_field($_GET['community_search']) : '';
?>

<section class="hero-bg min-h-[60vh] flex items-center justify-center relative">
    <div class="w-full max-w-6xl mx-auto px-6 text-center text-white">
        <h1 class="text-5xl md:text-6xl font-bold mb-6 leading-tight">Our Communities</h1>
        <p class="text-xl md:text-2xl mb-8 opacity-90">Discover exceptional neighborhoods throughout the Northeast where luxury meets lifestyle</p>
        <div class="relative max-w-2xl mx-auto">
            <form method="get" action="<?php echo esc_url(get_post_type_archive_link('community')); ?>">
                <input type="text" name="community_search" value="<?php echo esc_attr($search_query); ?>" placeholder="Search communities by name, city, or state" class="w-full px-6 py-4 pr-16 border-none rounded-2xl focus:outline-none focus:ring-2 focus:ring-primary text-gray-800 text-lg" id="communitySearch">
                <?php if ($current_state) : ?>
                    <input type="hidden" name="state" value="<?php echo esc_attr($current_state); ?>">
                <?php endif; ?>
                <button type="submit" class="absolute right-4 top-1/2 transform -translate-y-1/2 w-6 h-6 flex items-center justify-center">
                    <i class="ri-search-line text-gray-400 ri-lg"></i>
                </button>
            </form>
        </div>
    </div>
</section>

<section class="py-20 bg-white">
    <div class="max-w-7xl mx-auto px-6">
        <div class="flex flex-col md:flex-row justify-between items-center mb-12">
            <div>
                <h2 class="text-3xl md:text-4xl font-bold text-gray-900 mb-2">Featured Communities</h2>
                <p class="text-lg text-gray-600">Explore our carefully curated selection of premier neighborhoods</p>
            </div>
            <div class="flex flex-wrap gap-3 mt-6 md:mt-0">
                <?php
                // Get all states with communities
                $states = get_terms(array(
                    'taxonomy' => 'state',
                    'hide_empty' => true,
                ));

                // All Communities button
                $all_active = empty($current_state) ? 'active' : '';
                ?>
                <a href="<?php echo esc_url(get_post_type_archive_link('community')); ?>" 
                   class="filter-button <?php echo $all_active; ?> px-6 py-2 rounded-full font-medium !rounded-button whitespace-nowrap">
                    All Communities
                </a>
                
                <?php foreach ($states as $state) : 
                    $active_class = ($current_state === $state->slug) ? 'active' : '';
                    $filter_url = add_query_arg('state', $state->slug, get_post_type_archive_link('community'));
                    if ($search_query) {
                        $filter_url = add_query_arg('community_search', $search_query, $filter_url);
                    }
                ?>
                    <a href="<?php echo esc_url($filter_url); ?>" 
                       class="filter-button <?php echo $active_class; ?> px-6 py-2 rounded-full font-medium !rounded-button whitespace-nowrap">
                        <?php echo esc_html($state->name); ?>
                    </a>
                <?php endforeach; ?>
            </div>
        </div>

        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8" id="communitiesGrid">
            <?php
            // Set up query args
            $paged = (get_query_var('paged')) ? get_query_var('paged') : 1;
            $args = array(
                'post_type' => 'community',
                'posts_per_page' => 9,
                'paged' => $paged,
            );

            // Add state filter if set
            if ($current_state) {
                $args['tax_query'] = array(
                    array(
                        'taxonomy' => 'state',
                        'field' => 'slug',
                        'terms' => $current_state,
                    ),
                );
            }

            // Add search query if set
            if ($search_query) {
                $args['s'] = $search_query;
            }

            $communities_query = new WP_Query($args);

            if ($communities_query->have_posts()) :
                while ($communities_query->have_posts()) : $communities_query->the_post();
                    // Get custom fields
                    $avg_price = get_post_meta(get_the_ID(), 'community_avg_price', true);
                    $active_listings = get_post_meta(get_the_ID(), 'community_active_listings', true);
                    $location = get_post_meta(get_the_ID(), 'community_location', true);
                    $community_type = get_post_meta(get_the_ID(), 'community_type', true);
                    
                    // Get state term
                    $states = get_the_terms(get_the_ID(), 'state');
                    $state_name = $states ? $states[0]->name : '';
                    $state_slug = $states ? $states[0]->slug : '';
                    
                    // Set type badge color
                    $badge_colors = array(
                        'Featured' => 'bg-primary',
                        'Historic' => 'bg-orange-500',
                        'Resort Living' => 'bg-green-500',
                        'Academic' => 'bg-purple-500',
                        'Waterfront' => 'bg-blue-500',
                        'Lakefront' => 'bg-teal-500',
                        'Island Living' => 'bg-cyan-500',
                        'Country Club' => 'bg-indigo-500',
                        'Village Living' => 'bg-amber-500',
                    );
                    $badge_color = isset($badge_colors[$community_type]) ? $badge_colors[$community_type] : 'bg-gray-500';
                    ?>
                    <div class="community-card bg-white rounded-2xl overflow-hidden shadow-lg" data-category="<?php echo esc_attr($state_slug); ?>">
                        <div class="relative">
                            <?php if (has_post_thumbnail()) : ?>
                                <?php the_post_thumbnail('large', array('class' => 'w-full h-64 object-cover object-top')); ?>
                            <?php else : ?>
                                <img src="<?php echo esc_url(get_template_directory_uri() . '/assets/images/community-placeholder.jpg'); ?>" 
                                     alt="<?php the_title(); ?>" 
                                     class="w-full h-64 object-cover object-top">
                            <?php endif; ?>
                            <?php if ($community_type) : ?>
                                <div class="absolute top-4 left-4 <?php echo esc_attr($badge_color); ?> text-white px-3 py-1 rounded-full text-sm font-semibold">
                                    <?php echo esc_html($community_type); ?>
                                </div>
                            <?php endif; ?>
                        </div>
                        <div class="p-6">
                            <h3 class="text-2xl font-bold text-gray-900 mb-2"><?php the_title(); ?></h3>
                            <p class="text-gray-600 mb-4"><?php echo wp_trim_words(get_the_excerpt(), 25); ?></p>
                            <div class="flex items-center justify-between mb-6">
                                <div class="text-sm text-gray-500">
                                    <div class="flex items-center space-x-1 mb-1">
                                        <div class="w-4 h-4 flex items-center justify-center">
                                            <i class="ri-map-pin-line"></i>
                                        </div>
                                        <span><?php echo esc_html($location); ?>, <?php echo esc_html($state_name); ?></span>
                                    </div>
                                    <div class="flex items-center space-x-1">
                                        <div class="w-4 h-4 flex items-center justify-center">
                                            <i class="ri-home-line"></i>
                                        </div>
                                        <span><?php echo esc_html($active_listings); ?> Active Listings</span>
                                    </div>
                                </div>
                                <div class="text-right">
                                    <div class="text-sm text-gray-500">Avg. Price</div>
                                    <div class="text-2xl font-bold text-gray-900">$<?php echo esc_html($avg_price); ?></div>
                                </div>
                            </div>
                            <a href="<?php the_permalink(); ?>" class="w-full bg-gray-900 text-white py-3 rounded-lg hover:bg-gray-800 transition-colors font-semibold flex items-center justify-center space-x-2 whitespace-nowrap !rounded-button">
                                <span>View Community</span>
                                <div class="w-4 h-4 flex items-center justify-center">
                                    <i class="ri-arrow-right-line"></i>
                                </div>
                            </a>
                        </div>
                    </div>
                <?php endwhile; ?>
            <?php else : ?>
                <div class="col-span-full text-center py-12">
                    <p class="text-xl text-gray-600">No communities found matching your criteria.</p>
                </div>
            <?php endif; 
            wp_reset_postdata();
            ?>
        </div>
        
        <?php if ($communities_query->max_num_pages > 1) : ?>
        <div class="mt-12 flex justify-center">
            <?php
            echo paginate_links(array(
                'total' => $communities_query->max_num_pages,
                'current' => $paged,
                'prev_text' => '<i class="ri-arrow-left-line"></i>',
                'next_text' => '<i class="ri-arrow-right-line"></i>',
                'type' => 'list',
                'before_page_number' => '<span class="screen-reader-text">Page </span>',
            ));
            ?>
        </div>
        <?php endif; ?>
    </div>
</section>

<section class="py-20 bg-gray-50">
    <div class="max-w-4xl mx-auto px-6">
        <div class="text-center mb-16">
            <h2 class="text-4xl md:text-5xl font-bold text-gray-900 mb-6">Why Choose Northeast Communities?</h2>
            <p class="text-xl text-gray-600 leading-relaxed">The Northeast region offers an unparalleled combination of natural beauty, cultural richness, and economic opportunity, making it one of the most desirable places to call home in the United States.</p>
        </div>

        <div class="prose prose-lg max-w-none">
            <h3 class="text-2xl font-bold text-gray-900 mb-4">Rich History and Cultural Heritage</h3>
            <p class="text-gray-700 mb-6">From the cobblestone streets of Boston's Beacon Hill to the colonial charm of Connecticut's Gold Coast, Northeast communities are steeped in American history. These neighborhoods offer residents the unique opportunity to live among centuries-old architecture while enjoying modern amenities and conveniences.</p>

            <h3 class="text-2xl font-bold text-gray-900 mb-4">Educational Excellence</h3>
            <p class="text-gray-700 mb-6">The Northeast is home to some of the world's most prestigious educational institutions, including Harvard, MIT, Yale, and Dartmouth. Many of our featured communities are located in top-rated school districts, making them ideal for families seeking exceptional educational opportunities for their children.</p>

            <h3 class="text-2xl font-bold text-gray-900 mb-4">Four-Season Beauty</h3>
            <p class="text-gray-700 mb-6">Experience the stunning transformation of New England's landscapes throughout the year. From vibrant spring blooms to spectacular autumn foliage, snowy winter wonderlands to lush summer greenery, Northeast communities offer year-round natural beauty that enhances daily life.</p>

            <h3 class="text-2xl font-bold text-gray-900 mb-4">Economic Opportunities</h3>
            <p class="text-gray-700 mb-6">The Northeast corridor is a major economic powerhouse, home to thriving industries including technology, finance, healthcare, and biotechnology. Cities like Boston, New York, and Hartford offer abundant career opportunities while maintaining close proximity to peaceful suburban and rural communities.</p>

            <h3 class="text-2xl font-bold text-gray-900 mb-4">Outdoor Recreation</h3>
            <p class="text-gray-700 mb-6">Whether you're drawn to the pristine beaches of Cape Cod and Nantucket, the world-class skiing in Vermont's Green Mountains, or the hiking trails of New Hampshire's White Mountains, Northeast communities provide easy access to diverse recreational activities for every season and interest.</p>

            <div class="bg-white rounded-2xl p-8 mt-12">
                <h3 class="text-2xl font-bold text-gray-900 mb-4">Ready to Find Your Perfect Community?</h3>
                <p class="text-gray-700 mb-6">Our experienced team of real estate professionals knows these communities inside and out. We'll help you discover the neighborhood that perfectly matches your lifestyle, preferences, and investment goals.</p>
                <div class="flex flex-col sm:flex-row gap-4">
                    <a href="<?php echo esc_url(home_url('/contact')); ?>" class="bg-primary text-white px-8 py-3 rounded-lg hover:bg-blue-700 transition-colors font-semibold flex items-center justify-center space-x-2 !rounded-button whitespace-nowrap">
                        <span>Schedule a Community Tour</span>
                        <div class="w-4 h-4 flex items-center justify-center">
                            <i class="ri-calendar-line"></i>
                        </div>
                    </a>
                    <a href="<?php echo esc_url(home_url('/contact')); ?>" class="bg-gray-100 text-gray-900 px-8 py-3 rounded-lg hover:bg-gray-200 transition-colors font-semibold flex items-center justify-center space-x-2 !rounded-button whitespace-nowrap">
                        <span>Contact Our Team</span>
                        <div class="w-4 h-4 flex items-center justify-center">
                            <i class="ri-phone-line"></i>
                        </div>
                    </a>
                </div>
            </div>
        </div>
    </div>
</section>

<?php get_footer(); ?>