<?php

/*
Plugin Name: Custom Admin Theme
Plugin URI: https://mediatech.group
Description: A custom admin theme for your custom website by Media & Technology Group, LLC.
Author: Media & Technology Group, LLC
Version: 1.2.9
Author URI: https://mediatech.group
*/

defined( 'ABSPATH' ) || exit;

define('MTG_ADMIN_THEME_PLUGIN_VERSION', '1.2.9');
define('MTG_ADMIN_SETTINGS_PAGE', 'mtg-admin-settings');

if( ! class_exists( 'mtgUpdateChecker' ) ) {

	class mtgUpdateChecker{

		public $plugin_slug;
		public $version;
		public $cache_key;
		public $cache_allowed;

		public function __construct() {

			$this->plugin_slug = plugin_basename( __DIR__ );
			$this->version = MTG_ADMIN_THEME_PLUGIN_VERSION;
			$this->cache_key = 'mtg_custom_upd';
			$this->cache_allowed = false;

			add_filter( 'plugins_api', array( $this, 'info' ), 20, 3 );
			add_filter( 'site_transient_update_plugins', array( $this, 'update' ) );
			add_action( 'upgrader_process_complete', array( $this, 'purge' ), 10, 2 );

		}

		public function request(){

			$remote = get_transient( $this->cache_key );

			if( false === $remote || ! $this->cache_allowed ) {

				$remote = wp_remote_get( 
					add_query_arg( 
						array(
							'license_key' => urlencode( get_option( 'mtg_license_key' ) )
						), 
						'https://mediatech.group/updater/info.php'
					), 
					array(
						'timeout' => 10,
						'headers' => array(
							'Accept' => 'application/json'
						)
					)
				);

				if(
					is_wp_error( $remote )
					|| 200 !== wp_remote_retrieve_response_code( $remote )
					|| empty( wp_remote_retrieve_body( $remote ) )
				) {
					return false;
				}

				set_transient( $this->cache_key, $remote, DAY_IN_SECONDS );

			}

			$remote = json_decode( wp_remote_retrieve_body( $remote ) );

			return $remote;

		}


		function info( $res, $action, $args ) {

			// print_r( $action );
			// print_r( $args );

			// do nothing if you're not getting plugin information right now
			if( 'plugin_information' !== $action ) {
				return $res;
			}

			// do nothing if it is not our plugin
			if( $this->plugin_slug !== $args->slug ) {
				return $res;
			}

			// get updates
			$remote = $this->request();

			if( ! $remote ) {
				return $res;
			}

			$res = new stdClass();

			$res->name = $remote->name;
			$res->slug = $remote->slug;
			$res->version = $remote->version;
			$res->tested = $remote->tested;
			$res->requires = $remote->requires;
			$res->author = $remote->author;
			$res->author_profile = $remote->author_profile;
			$res->download_link = $remote->download_url;
			$res->trunk = $remote->download_url;
			$res->requires_php = $remote->requires_php;
			$res->last_updated = $remote->last_updated;

			$res->sections = array(
				'description' => $remote->sections->description,
				'installation' => $remote->sections->installation,
				'changelog' => $remote->sections->changelog
			);

			if( ! empty( $remote->banners ) ) {
				$res->banners = array(
					'low' => $remote->banners->low,
					'high' => $remote->banners->high
				);
			}

			return $res;

		}

		public function update( $transient ) {

			if ( empty($transient->checked ) ) {
				return $transient;
			}

			$remote = $this->request();

			if(
				$remote
				&& version_compare( $this->version, $remote->version, '<' )
				&& version_compare( $remote->requires, get_bloginfo( 'version' ), '<=' )
				&& version_compare( $remote->requires_php, PHP_VERSION, '<' )
			) {
				$res = new stdClass();
				$res->slug = $this->plugin_slug;
				$res->plugin = plugin_basename( __FILE__ );
				$res->new_version = $remote->version;
				$res->tested = $remote->tested;
				$res->package = $remote->download_url;

				$transient->response[ $res->plugin ] = $res;

	    }

			return $transient;

		}

		public function purge( $upgrader, $options ){

			if (
				$this->cache_allowed
				&& 'update' === $options['action']
				&& 'plugin' === $options[ 'type' ]
			) {
				// just clean the cache when new plugin version is installed
				delete_transient( $this->cache_key );
			}

		}


	}

	new mtgUpdateChecker();

}

add_action( 'in_plugin_update_message-mediatech-admin-theme/mediatech-admin-theme.php', 'mtg_update_message', 10, 2 );
function mtg_update_message( $plugin_info_array, $plugin_info_object ) {
	if( empty( $plugin_info_array[ 'package' ] ) ) {
		echo ' Please renew your license to update. You can change your license key in Settings > General';
	}
}

// Add a menu item in the settings menu
add_action('admin_menu', 'mtg_admin_settings_menu');
function mtg_admin_settings_menu() {
    add_menu_page('Media & Technology Group', 'Media & Technology Group', 'manage_options', MTG_ADMIN_SETTINGS_PAGE, 'mtg_admin_settings_page');
}

// Create the settings page for the plugin
function mtg_admin_settings_page() {
    ?>
    <div class="wrap">
        <h2>Media & Technoloy Group Admin Settings</h2>
        <form method="post" action="options.php">
            <?php settings_fields('mtg_admin_settings_options'); ?>
            <?php do_settings_sections('mtg_admin_settings'); ?>
            <?php submit_button(); ?>
        </form>
    </div>
	<div>
		<h2>Website Manager Role Capabilities</h2>
		<?php var_dump(km_get_user_capabilities(2)); ?>
	</div>
	<div><h2>Menu Pages for Admin</h2>
		<?php var_dump($GLOBALS[ 'menu' ]); ?>
	</div>
    <?php
	
}

// Register the plugin settings and options
add_action('admin_init', 'mtg_admin_settings_settings');
function mtg_admin_settings_settings() {
    register_setting('mtg_admin_settings_options', 'mtg_license_key');
    add_settings_section('mtg_admin_settings_section', 'MTG License Key', 'mtg_admin_settings_section_text', 'mtg_admin_settings');
    add_settings_field('mtg_license_key', 'License Key', 'mtg_admin_settings_setting', 'mtg_admin_settings', 'mtg_admin_settings_section');
}

// Display the text for the settings section
function mtg_admin_settings_section_text() {
    echo '<p>Enter your Media & Technology Group license key below. This unlocks a variety of customizations, performance improvements, and premium chat support directly from the backend of your website!</p>';
}

// Display the input field for the license key option
function mtg_admin_settings_setting() {
    $license = get_option('mtg_license_key');
	$status  = get_option('mtg_license_status');
    echo "<input type='text' name='mtg_license_key' value='" . esc_attr($license) . "' />";
	
	if( $status !== false && $status == 'valid' ) {
		echo '<span style="color:green;">' . _e('active') . '</span>';
		echo wp_nonce_field('mtg_license_nonce', 'mtg_license_nonce');
		echo '<input type="submit" class="button-secondary" name="mtg_license_deactivate" value="' . _e('Deactivate License') . '"';
	}
	else {
		echo wp_nonce_field('mtg_license_nonce', 'mtg_license_nonce');
		echo '<input type="submit" class="button-secondary" name="mtg_license_activate" value="' . _e('Activate License') . '"/>';
	}
}

function mtg_activate_license() {

	// listen for our activate button to be clicked
	if( isset( $_POST['mtg_license_activate'] ) ) {

		// run a quick security check
	 	if( ! check_admin_referer( 'mtg_license_nonce', 'mtg_license_nonce' ) )
			return; // get out if we didn't click the Activate button

		// retrieve the license from the database
		$license = trim( get_option('mtg_license_key') );


		// data to send in our API request
		$api_params = array(
			'mtg_action' => 'activate_license',
			'license'    => $license,
			'url'        => home_url()
		);

		// Call the custom API.
		$response = wp_remote_post( 'https://mediatech.group/updater/info.php', array( 'timeout' => 15, 'sslverify' => false, 'body' => $api_params ) );

		// make sure the response came back okay
		if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {

			$message =  ( is_wp_error( $response ) && ! empty( $response->get_error_message() ) ) ? $response->get_error_message() : __( 'An error occurred, please try again.' );

		} else {

			$license_data = json_decode( wp_remote_retrieve_body( $response ) );

			if ( false === $license_data->success ) {

				switch( $license_data->error ) {

					case 'expired' :

						$message = sprintf(
							__( 'Your license key expired on %s.' ),
							date_i18n( get_option( 'date_format' ), strtotime( $license_data->expires, current_time( 'timestamp' ) ) )
						);
						break;

					case 'revoked' :

						$message = __( 'Your license key has been disabled.' );
						break;

					case 'missing' :

						$message = __( 'Invalid license.' );
						break;

					case 'invalid' :
					case 'site_inactive' :

						$message = __( 'Your license is not active for this URL.' );
						break;

					case 'item_name_mismatch' :

						$message = sprintf( __( 'This appears to be an invalid license key for %s.' ), MTG_ADMIN_SETTINGS_PAGE );
						break;

					case 'no_activations_left':

						$message = __( 'Your license key has reached its activation limit.' );
						break;

					default :

						$message = __( 'An error occurred, please try again.' );
						break;
				}

			}

		}

		// Check if anything passed on a message constituting a failure
		if ( ! empty( $message ) ) {
			$base_url = admin_url( 'plugins.php?page=' . MTG_ADMIN_SETTINGS_PAGE );
			$redirect = add_query_arg( array( 'sl_activation' => 'false', 'message' => urlencode( $message ) ), $base_url );

			wp_redirect( $redirect );
			exit();
		}

		// $license_data->license will be either "valid" or "invalid"

		update_option( 'edd_sample_license_status', $license_data->license );
		wp_redirect( admin_url( 'plugins.php?page=' . MTG_ADMIN_SETTINGS_PAGE ) );
		exit();
	}
}
add_action('admin_init', 'mtg_activate_license');

function mtg_license_admin_notices() {
	if ( isset( $_GET['sl_activation'] ) && ! empty( $_GET['message'] ) ) {

		switch( $_GET['sl_activation'] ) {

			case 'false':
				$message = urldecode( $_GET['message'] );
				?>
				<div class="error">
					<p><?php echo $message; ?></p>
				</div>
				<?php
				break;

			case 'true':
			default:
				$message = 'Plugin successfully activated! Thanks for being an important part of our family.';
				break;

		}
	}
}
add_action( 'admin_notices', 'mtg_license_admin_notices' );

function mtg_admin_theme_enqueue() {
    wp_enqueue_style('mediatech_admin_theme_styles', plugins_url('styles.css', __FILE__), array(), MTG_ADMIN_THEME_PLUGIN_VERSION, false);
	wp_enqueue_script('mediatech_admin_theme_scripts', plugins_url('scripts.js', __FILE__), array(), MTG_ADMIN_THEME_PLUGIN_VERSION, true);
}
add_action('admin_enqueue_scripts', 'mtg_admin_theme_enqueue');
add_action('login_enqueue_scripts', 'mtg_admin_theme_enqueue');
add_action('login_head', 'mtg_admin_theme_enqueue');

function mtg_login_js() {
	wp_enqueue_script( 'mediatech_login_js', plugins_url('login.js', __FILE__), array(), MTG_ADMIN_THEME_PLUGIN_VERSION, true);
}
add_action('login_enqueue_scripts', 'mtg_login_js');
 
add_filter('update_footer', 'right_admin_footer_text_output', 11);
function right_admin_footer_text_output($text) {
	$logo_url = plugins_url('/assets/mt-group-logo-white-1000x148.png', __FILE__);
    $text = '<div id="mtg-admin-footer-text">';
	$text .= '<span class="mtg-admin-footer-text-top">A website made just for you by <a href="https://mediatech.group" target="blank"><img src="' . $logo_url . '" alt="Media & Technology Group, LLC" /></a></span><br />';
	$text .= 'For support, call <a href="tel:+17172563886‬">(717) 256-3886</a>, email <a href="mailto:support@mediatech.group">support@mediatech.group</a>, or use the chat located in the bottom-right!';
	$text .= '</div>';
	$text .= '<script type="text/javascript">var $zoho=$zoho || {};$zoho.salesiq = $zoho.salesiq || {widgetcode:"f113ceb939d559a4b3d44373860f0336508886057c56238b8a60a559056f5cd3066d5c72205787973d7542a2716d72ae", values:{},ready:function(){}};var d=document;s=d.createElement("script");s.type="text/javascript";s.id="zsiqscript";s.defer=true;s.src="https://salesiq.zoho.com/widget";t=d.getElementsByTagName("script")[0];t.parentNode.insertBefore(s,t);d.write("<div id=\'zsiqwidget\'></div>");</script>';
    return $text;
}

function mtg_dashboard_widgets() {
	remove_meta_box('dashboard_right_now', 'dashboard', 'normal' );
	remove_meta_box('dashboard_recent_comments', 'dashboard', 'normal' );
	remove_meta_box('dashboard_incoming_links', 'dashboard', 'normal' );
	remove_meta_box('dashboard_plugins', 'dashboard', 'normal' );
	remove_meta_box('dashboard_quick_press', 'dashboard', 'side' );
	remove_meta_box('dashboard_recent_drafts', 'dashboard', 'side' );
	remove_meta_box('dashboard_primary', 'dashboard', 'side' );
	remove_meta_box('dashboard_secondary', 'dashboard', 'side' );
	remove_meta_box('e-dashboard-overview', 'dashboard', 'normal');
	remove_meta_box('dashboard_site_health', 'dashboard', 'normal');
	remove_meta_box('dashboard_activity', 'dashboard', 'normal');
}
// 
add_action( 'wp_dashboard_setup', 'mtg_dashboard_widgets' );

function mtg_add_dashboard_widgets() {
	wp_add_dashboard_widget(
		'mtg_support_dashboard_widget', // Widget slug
		'Media & Technology Group Support', // Widget title
		'mtg_support_dashboard_widget_function' // Function name to display the widget
	);
}

add_action( 'wp_dashboard_setup', 'mtg_add_dashboard_widgets' );

function mtg_support_dashboard_widget_function() {
	echo '<h4>Hello from Media & Technology Group, LLC!</h4>';
	echo '<p>As always, thank you for choosing to be a part of our family. In an effort to demonstrate how truly greatful we are for your partnership, we\'ve invested considerable effort into making it easy for you to get in touch for support as it is needed. You can always give a call or text <a href="tel:+17172563886">(717) 256-3886</a>, email <a href="mailto:support@mediatech.group">support@mediatech.group</a>, or use the chat system built directly into the backend of your website (open the chat from the icon on the bottom-right of the screen)!</p>';
	echo '<p>Thanks again and please do let us know without hesitation how we can serve you.</p>';
}

if( is_admin() ){
    remove_action( 'admin_color_scheme_picker', 'admin_color_scheme_picker' );
    add_action( 'personal_options', 'prefix_hide_personal_options' );	
}
 
function prefix_hide_personal_options() {
  ?>
    <script type="text/javascript">
        jQuery( document ).ready(function( $ ){
            $( '#your-profile .form-table:first, .user-description-wrap, .user-profile-picture, h2, .user-pinterest-wrap, .user-myspace-wrap, .user-soundcloud-wrap, .user-tumblr-wrap, .user-wikipedia-wrap' ).remove();
        } );
    </script>
  <?php
}

function mtg_client_role() {
	if ( get_role( 'website_manager' ) === null ) {
		$capabilities = get_role( 'administrator' )->capabilities;
		add_role('website_manager','Website Manager', $capabilities);
	}
	//remove_role('website_manager');
}
add_action('init', 'mtg_client_role');

function mtg_remove_custom_admin_capabilities() {
    $mtg_client_role = get_role( 'website_manager' );
    $unwanted_capabilities = array(
        'activate_plugins',
        'edit_theme_options',
        'edit_dashboard',
        'manage_options',
        'switch_themes',
        'delete_plugins',
        'delete_themes',
        'edit_plugins',
        'edit_themes',
    );
    foreach ( $unwanted_capabilities as $capability ) {
        $mtg_client_role->remove_cap( $capability );
    }
}
add_action( 'init', 'mtg_remove_custom_admin_capabilities' );

function mtg_remove_plugins_menu_item() {
    $user = wp_get_current_user();
    $user_roles = $user->roles;
    if (in_array('website_manager', $user_roles)) {
        remove_submenu_page('index.php', 'update-core.php');
    }
	
	if ( ! current_user_can( 'activate_plugins' ) ) {
        remove_menu_page( 'plugins.php' );
		remove_submenu_page( 'plugins.php', 'plugin-editor.php' );
    }
	if ( ! current_user_can( 'edit_theme_options' ) ) {
        remove_menu_page( 'themes.php' );
		remove_submenu_page( 'themes.php', 'theme-editor.php' );
    }
	if ( ! current_user_can( 'manage_options' ) ) {
        remove_menu_page( 'options-general.php' );
    }
}
add_action( 'admin_menu', 'mtg_remove_plugins_menu_item' );

function login_logo_url() {
    return home_url();
}
add_filter( 'login_headerurl', 'login_logo_url' );

function km_get_user_capabilities( $user = null ) {

	$user = $user ? new WP_User( $user ) : wp_get_current_user();

	return array_keys( $user->allcaps );
}