/**
 * Agent Single Page JavaScript
 * Handles interactive features for the single agent template
 */

(function($) {
    'use strict';

    // Wait for DOM to be ready
    $(document).ready(function() {
        initTestimonialSlider();
        initConsultationButtons();
        initContactForm();
        initSmoothScroll();
        initPropertyCards();
    });

    /**
     * Initialize testimonial slider
     */
    function initTestimonialSlider() {
        const $track = $('#testimonialTrack');
        const $dots = $('.testimonial-dot');
        
        if (!$track.length || !$dots.length) return;
        
        let currentSlide = 0;
        const totalSlides = $dots.length;
        let autoplayInterval;

        function updateSlider() {
            const offset = currentSlide * -100;
            $track.css('transform', `translateX(${offset}%)`);
            
            $dots.removeClass('bg-primary').addClass('bg-gray-300');
            $dots.eq(currentSlide).removeClass('bg-gray-300').addClass('bg-primary');
        }

        // Handle dot clicks
        $dots.on('click', function() {
            currentSlide = $(this).data('slide');
            updateSlider();
            resetAutoplay();
        });

        // Touch/swipe support
        let touchStartX = 0;
        let touchEndX = 0;

        $track.on('touchstart', function(e) {
            touchStartX = e.changedTouches[0].screenX;
        });

        $track.on('touchend', function(e) {
            touchEndX = e.changedTouches[0].screenX;
            handleSwipe();
        });

        function handleSwipe() {
            if (touchEndX < touchStartX - 50) {
                // Swipe left
                currentSlide = (currentSlide + 1) % totalSlides;
                updateSlider();
                resetAutoplay();
            }
            if (touchEndX > touchStartX + 50) {
                // Swipe right
                currentSlide = (currentSlide - 1 + totalSlides) % totalSlides;
                updateSlider();
                resetAutoplay();
            }
        }

        // Autoplay
        function startAutoplay() {
            autoplayInterval = setInterval(function() {
                currentSlide = (currentSlide + 1) % totalSlides;
                updateSlider();
            }, 5000);
        }

        function resetAutoplay() {
            clearInterval(autoplayInterval);
            startAutoplay();
        }

        // Initialize
        updateSlider();
        startAutoplay();

        // Pause on hover
        $('.testimonial-slider').on('mouseenter', function() {
            clearInterval(autoplayInterval);
        }).on('mouseleave', function() {
            startAutoplay();
        });
    }

    /**
     * Initialize consultation button handlers
     */
    function initConsultationButtons() {
        $('[data-agent-consultation]').on('click', function(e) {
            e.preventDefault();
            
            // Smooth scroll to contact form
            const $contactSection = $('#agent-contact-section, .agent-contact-form').closest('section');
            if ($contactSection.length) {
                $('html, body').animate({
                    scrollTop: $contactSection.offset().top - 100
                }, 800);
                
                // Focus on first form field after scroll
                setTimeout(function() {
                    $contactSection.find('input:first').focus();
                }, 900);
            }
        });
    }

    /**
     * Initialize contact form handling
     */
    function initContactForm() {
        const $form = $('.agent-contact-form');
        
        if (!$form.length) return;

        $form.on('submit', function(e) {
            const $submitBtn = $(this).find('button[type="submit"]');
            const originalText = $submitBtn.text();
            
            // Update button state
            $submitBtn.prop('disabled', true)
                     .html('<i class="ri-loader-4-line ri-spin"></i> Sending...');
            
            // If not using Contact Form 7, handle the submission via AJAX
            if (!$(this).hasClass('wpcf7-form')) {
                e.preventDefault();
                
                $.ajax({
                    url: elite_realty_ajax.ajax_url,
                    type: 'POST',
                    data: $(this).serialize(),
                    success: function(response) {
                        if (response.success) {
                            // Show success message
                            showFormMessage($form, 'success', 'Thank you! Your message has been sent successfully.');
                            $form[0].reset();
                        } else {
                            showFormMessage($form, 'error', response.data || 'An error occurred. Please try again.');
                        }
                    },
                    error: function() {
                        showFormMessage($form, 'error', 'An error occurred. Please try again later.');
                    },
                    complete: function() {
                        $submitBtn.prop('disabled', false).text(originalText);
                    }
                });
            }
        });

        // Handle Contact Form 7 events
        if (typeof wpcf7 !== 'undefined') {
            $(document).on('wpcf7:submit', function(e) {
                const $form = $(e.target);
                const $submitBtn = $form.find('input[type="submit"]');
                $submitBtn.prop('disabled', false);
            });
        }
    }

    /**
     * Show form message
     */
    function showFormMessage($form, type, message) {
        const alertClass = type === 'success' ? 'bg-green-50 text-green-800' : 'bg-red-50 text-red-800';
        const iconClass = type === 'success' ? 'ri-check-line' : 'ri-error-warning-line';
        
        const $message = $(`
            <div class="form-message ${alertClass} p-4 rounded-lg mb-4 flex items-start space-x-3">
                <i class="${iconClass} text-xl flex-shrink-0 mt-0.5"></i>
                <p>${message}</p>
            </div>
        `);
        
        $form.prepend($message);
        
        // Remove message after 5 seconds
        setTimeout(function() {
            $message.fadeOut(300, function() {
                $(this).remove();
            });
        }, 5000);
    }

    /**
     * Initialize smooth scroll for anchor links
     */
    function initSmoothScroll() {
        $('a[href^="#"]').on('click', function(e) {
            const target = $(this.getAttribute('href'));
            if (target.length) {
                e.preventDefault();
                $('html, body').animate({
                    scrollTop: target.offset().top - 100
                }, 800);
            }
        });
    }

    /**
     * Initialize property card interactions
     */
    function initPropertyCards() {
        // Add loading state for property images
        $('.listing-card img, .transaction-card img').on('load', function() {
            $(this).addClass('fade-in-up');
        }).on('error', function() {
            $(this).attr('src', elite_realty_ajax.template_url + '/assets/images/property-placeholder.jpg');
        });

        // Add hover effect for touch devices
        $('.listing-card, .transaction-card').on('touchstart', function() {
            $(this).addClass('hover');
        }).on('touchend', function() {
            $(this).removeClass('hover');
        });
    }

    /**
     * Lazy load images
     */
    function initLazyLoad() {
        if ('IntersectionObserver' in window) {
            const imageObserver = new IntersectionObserver(function(entries, observer) {
                entries.forEach(function(entry) {
                    if (entry.isIntersecting) {
                        const image = entry.target;
                        image.src = image.dataset.src;
                        image.classList.add('fade-in-up');
                        imageObserver.unobserve(image);
                    }
                });
            });

            document.querySelectorAll('img[data-src]').forEach(function(img) {
                imageObserver.observe(img);
            });
        }
    }

    // Initialize lazy loading
    initLazyLoad();

    /**
     * Handle responsive behavior
     */
    function handleResponsive() {
        const isMobile = window.innerWidth < 768;
        
        // Adjust testimonial slider for mobile
        if (isMobile) {
            $('.testimonial-card').css('flex', '0 0 100%');
        } else {
            $('.testimonial-card').css('flex', '0 0 33.333%');
        }
    }

    // Handle window resize
    let resizeTimer;
    $(window).on('resize', function() {
        clearTimeout(resizeTimer);
        resizeTimer = setTimeout(handleResponsive, 250);
    });

    // Initial responsive check
    handleResponsive();

})(jQuery);